#!/usr/bin/env python3
"""
Script auxiliar para criar arquivo markdown com todas as transcrições
"""
import sys
import os
from pathlib import Path
from datetime import datetime

def create_markdown_from_texts(base_dir):
    """Criar arquivo markdown com todos os textos"""
    base_path = Path(base_dir)
    texts_dir = base_path / 'textos'
    markdown_path = base_path / 'copy_bank_completo.md'
    
    if not texts_dir.exists():
        print(f"Diretório de textos não encontrado: {texts_dir}")
        return
    
    with open(markdown_path, 'w', encoding='utf-8') as md_file:
        md_file.write("# Copy Bank - Campanha Milla Borges\n\n")
        md_file.write(f"Compilado em: {datetime.now().strftime('%d/%m/%Y às %H:%M')}\n\n")
        md_file.write("---\n\n")
        
        # Coletar e ordenar arquivos
        text_files = sorted(texts_dir.glob('*.txt'))
        
        if text_files:
            print(f"Compilando {len(text_files)} arquivos de texto...")
            
            for i, text_file in enumerate(text_files, 1):
                try:
                    with open(text_file, 'r', encoding='utf-8') as f:
                        content = f.read()
                    
                    # Adicionar ao markdown
                    md_file.write(content)
                    md_file.write("\n\n---\n\n")
                    
                    print(f"  [{i}/{len(text_files)}] {text_file.name}")
                    
                except Exception as e:
                    print(f"  Erro ao ler {text_file.name}: {e}")
        else:
            md_file.write("*Nenhuma transcrição encontrada no diretório de textos.*\n\n")
        
        md_file.write("\n## Fim do Copy Bank\n")
    
    print(f"\nCopy bank completo salvo em: {markdown_path}")
    print(f"Tamanho: {markdown_path.stat().st_size / 1024:.1f} KB")

def main():
    if len(sys.argv) != 2:
        print("Uso: python create_markdown.py <diretório>")
        sys.exit(1)
    
    base_dir = sys.argv[1]
    
    if not os.path.exists(base_dir):
        print(f"Erro: Diretório '{base_dir}' não encontrado.")
        sys.exit(1)
    
    create_markdown_from_texts(base_dir)

if __name__ == "__main__":
    main()