#!/usr/bin/env python3
"""
Gerador de galeria com múltiplos templates
"""
import sys
import os
from pathlib import Path
from generate_gallery import GalleryGenerator
from jinja2 import Template

def generate_gallery_with_template(base_dir, template_name):
    """Gerar galeria com template específico"""
    generator = GalleryGenerator(base_dir)
    generator.collect_files()
    
    # Adicionar link do markdown se existir
    generator.data['markdown_link'] = generator.get_markdown_download_link()
    
    # Escolher template
    template_file = {
        'feed': 'gallery_template.html',
        'cards': 'gallery_template_cards.html',
        'masonry': 'gallery_template_masonry.html',
        'sidebar': 'gallery_template_sidebar.html',
        'minimal': 'gallery_template_minimal.html',
        'timeline': 'gallery_template_timeline.html'
    }.get(template_name, 'gallery_template.html')
    
    # Ler template
    template_path = Path(__file__).parent / template_file
    with open(template_path, 'r', encoding='utf-8') as f:
        template_html = f.read()
    
    # Renderizar
    template = Template(template_html)
    html_content = template.render(**generator.data)
    
    # Salvar com nome específico
    output_name = f'galeria_campanha_{template_name}.html'
    output_path = generator.base_dir / output_name
    with open(output_path, 'w', encoding='utf-8') as f:
        f.write(html_content)
    
    print(f"✅ Galeria '{template_name}' gerada: {output_path}")
    return output_path

def main():
    if len(sys.argv) < 2:
        print("Uso: python generate_gallery_multi.py <diretório> [template]")
        print("\nTemplates disponíveis:")
        print("  feed     - Layout estilo feed de redes sociais (padrão)")
        print("  cards    - Layout dark mode com cards modernos")
        print("  masonry  - Layout estilo Pinterest/editorial")
        print("  sidebar  - Dashboard com menu lateral")
        print("  minimal  - Grid minimalista estilo galeria de arte")
        print("  timeline - Timeline vertical com animações")
        print("\nExemplos:")
        print("  python generate_gallery_multi.py lmb01/")
        print("  python generate_gallery_multi.py lmb01/ cards")
        print("  python generate_gallery_multi.py lmb01/ all")
        sys.exit(1)
    
    base_dir = sys.argv[1]
    template_name = sys.argv[2] if len(sys.argv) > 2 else 'feed'
    
    if not os.path.exists(base_dir):
        print(f"❌ Erro: Diretório '{base_dir}' não encontrado.")
        sys.exit(1)
    
    if template_name == 'all':
        # Gerar todas as versões
        print("🎨 Gerando todas as versões da galeria...\n")
        for tpl in ['feed', 'cards', 'masonry', 'sidebar', 'minimal', 'timeline']:
            generate_gallery_with_template(base_dir, tpl)
        print(f"\n✨ Todas as galerias foram geradas em: {base_dir}")
    else:
        # Gerar versão específica
        if template_name not in ['feed', 'cards', 'masonry', 'sidebar', 'minimal', 'timeline']:
            print(f"❌ Template '{template_name}' não encontrado.")
            print("   Use: feed, cards, masonry, sidebar, minimal ou timeline")
            sys.exit(1)
        
        gallery_path = generate_gallery_with_template(base_dir, template_name)
        print(f"\n🌐 Abra o arquivo para visualizar: {gallery_path}")

if __name__ == "__main__":
    main()