#!/usr/bin/env python3
"""
Script principal para processar toda a campanha da Milla Borges
Executa em sequência:
1. Organização dos arquivos por tipo
2. Transcrição de vídeos, imagens e PDFs
3. Geração da galeria HTML
"""

import sys
import os
import subprocess
from pathlib import Path
import time

def run_command(command, description, timeout_seconds=1800):
    """Executar comando e mostrar progresso
    
    Args:
        command: Comando a executar
        description: Descrição do comando
        timeout_seconds: Tempo limite em segundos (padrão: 30 minutos)
    """
    print(f"\n{'='*60}")
    print(f"🔄 {description}")
    print(f"{'='*60}")
    
    try:
        # Executar com timeout
        result = subprocess.run(
            command, 
            shell=True, 
            check=True, 
            capture_output=True, 
            text=True,
            timeout=timeout_seconds
        )
        if result.stdout:
            print(result.stdout)
        return True
    except subprocess.TimeoutExpired:
        print(f"⏱️ Tempo limite excedido ({timeout_seconds//60} minutos)")
        print(f"   O processamento pode continuar em segundo plano.")
        return False
    except subprocess.CalledProcessError as e:
        print(f"❌ Erro ao executar: {description}")
        if e.stdout:
            print(f"Saída: {e.stdout}")
        if e.stderr:
            print(f"Erro: {e.stderr}")
        return False

def check_dependencies():
    """Verificar se todas as dependências estão instaladas"""
    print("🔍 Verificando dependências...")
    
    # Verificar Python
    try:
        subprocess.run(["python3", "--version"], check=True, capture_output=True)
        print("✅ Python3 instalado")
    except:
        print("❌ Python3 não encontrado")
        return False
    
    # Verificar Tesseract
    try:
        subprocess.run(["tesseract", "--version"], check=True, capture_output=True)
        print("✅ Tesseract instalado")
    except:
        print("❌ Tesseract não encontrado")
        print("   Por favor, instale:")
        print("   macOS: brew install tesseract tesseract-lang")
        print("   Ubuntu: sudo apt-get install tesseract-ocr tesseract-ocr-por")
        return False
    
    return True

def main():
    if len(sys.argv) != 2:
        print("Uso: python processar_campanha.py <diretório>")
        print("\nExemplo: python processar_campanha.py lmb01/")
        sys.exit(1)
    
    directory = sys.argv[1]
    
    if not os.path.exists(directory):
        print(f"❌ Erro: Diretório '{directory}' não encontrado.")
        sys.exit(1)
    
    # Verificar dependências
    if not check_dependencies():
        print("\n❌ Dependências faltando. Por favor, instale antes de continuar.")
        sys.exit(1)
    
    print(f"\n🚀 Processando campanha no diretório: {directory}")
    print(f"   Isso pode demorar vários minutos...")
    
    # Usar o ambiente virtual
    venv_python = "venv/bin/python"
    
    # Etapa 1: Organizar arquivos
    if not run_command(
        f"{venv_python} organizar_campanha.py '{directory}'",
        "Organizando arquivos por tipo",
        timeout_seconds=300  # 5 minutos
    ):
        print("❌ Falha na organização dos arquivos")
        sys.exit(1)
    
    time.sleep(2)
    
    # Etapa 2: Transcrever conteúdo
    print("\n⚠️  ATENÇÃO: A transcrição pode demorar MUITO tempo!")
    print("   - Vídeos serão transcritos com Whisper (pode levar vários minutos por vídeo)")
    print("   - Imagens serão processadas com OCR")
    print("   - PDFs terão texto extraído")
    print("   - Todos os textos serão corrigidos com IA")
    print("   - Tempo limite: 30 minutos")
    print("\n💡 Dica: Você pode executar o transcriber.py separadamente se precisar de mais tempo.")
    
    if not run_command(
        f"{venv_python} transcriber.py '{directory}'",
        "Transcrevendo e extraindo textos",
        timeout_seconds=1800  # 30 minutos
    ):
        print("\n⚠️  O processamento excedeu o limite de 30 minutos.")
        print("   Para continuar, execute manualmente:")
        print(f"   source venv/bin/activate && python transcriber.py '{directory}'")
        print("\n   Continuando com a geração da galeria...")
    
    time.sleep(2)
    
    # Etapa 3: Gerar galeria HTML
    if not run_command(
        f"{venv_python} generate_gallery.py '{directory}'",
        "Gerando galeria HTML",
        timeout_seconds=60  # 1 minuto
    ):
        print("❌ Falha ao gerar galeria")
        sys.exit(1)
    
    # Resumo final
    print(f"\n{'='*60}")
    print("✨ PROCESSAMENTO CONCLUÍDO!")
    print(f"{'='*60}")
    print(f"\n📁 Arquivos organizados em:")
    print(f"   - {directory}/videos/")
    print(f"   - {directory}/imagens/")
    print(f"   - {directory}/textos/")
    print(f"   - {directory}/pdfs/")
    print(f"\n📝 Transcrições salvas em:")
    print(f"   - {directory}/textos/")
    print(f"\n🌐 Galeria HTML disponível em:")
    print(f"   - {directory}/galeria_campanha.html")
    print(f"\n📊 Metadados salvos em:")
    print(f"   - {directory}/campaign_metadata.json")
    
    # Abrir galeria no navegador (opcional)
    gallery_path = Path(directory) / "galeria_campanha.html"
    if gallery_path.exists():
        print(f"\n🌐 Abrindo galeria no navegador...")
        try:
            if sys.platform == "darwin":  # macOS
                subprocess.run(["open", str(gallery_path)])
            elif sys.platform == "linux":
                subprocess.run(["xdg-open", str(gallery_path)])
            elif sys.platform == "win32":
                subprocess.run(["start", str(gallery_path)], shell=True)
        except:
            print("   Não foi possível abrir automaticamente.")
            print(f"   Abra manualmente: {gallery_path}")

if __name__ == "__main__":
    main()