#!/usr/bin/env python3
"""
Script para processar apenas as transcrições sem limite de tempo
Útil quando o processamento completo excede o timeout
"""

import sys
import os
import subprocess
from pathlib import Path

def main():
    if len(sys.argv) != 2:
        print("Uso: python processar_transcricoes.py <diretório>")
        print("\nExemplo: python processar_transcricoes.py lmb01/")
        sys.exit(1)
    
    directory = sys.argv[1]
    
    if not os.path.exists(directory):
        print(f"❌ Erro: Diretório '{directory}' não encontrado.")
        sys.exit(1)
    
    print(f"\n🚀 Processando transcrições em: {directory}")
    print("=" * 60)
    print("⚠️  ATENÇÃO: Este processo pode demorar MUITO tempo!")
    print("   - Vídeos serão transcritos com Whisper")
    print("   - Imagens serão processadas com OCR")
    print("   - PDFs terão texto extraído")
    print("   - Todos os textos serão corrigidos com IA")
    print("   - SEM LIMITE DE TEMPO - o processo continuará até concluir")
    print("=" * 60)
    print("\n💡 Você pode interromper com Ctrl+C a qualquer momento")
    print("   O progresso será salvo e você pode continuar depois\n")
    
    # Usar o ambiente virtual
    venv_python = "venv/bin/python"
    
    try:
        # Executar transcriber sem timeout
        subprocess.run(
            [venv_python, "transcriber.py", directory],
            check=True
        )
        
        print("\n✅ Transcrições processadas com sucesso!")
        
        # Criar arquivo markdown compilado
        print("\n📝 Criando arquivo markdown compilado...")
        subprocess.run(
            [venv_python, "create_markdown.py", directory],
            check=True
        )
        
        # Regenerar galeria
        print("\n🌐 Regenerando galeria HTML...")
        subprocess.run(
            [venv_python, "generate_gallery.py", directory],
            check=True
        )
        
        print("\n✨ PROCESSAMENTO CONCLUÍDO!")
        print(f"\n📄 Copy bank completo: {directory}/copy_bank_completo.md")
        print(f"🌐 Galeria HTML: {directory}/galeria_campanha.html")
        
    except subprocess.CalledProcessError as e:
        print(f"\n❌ Erro durante o processamento: {e}")
        print("   Verifique os logs acima para mais detalhes")
        sys.exit(1)
    except KeyboardInterrupt:
        print("\n\n⚠️  Processamento interrompido pelo usuário")
        print("   Para continuar, execute novamente este script")
        print("   O progresso foi salvo")
        sys.exit(0)

if __name__ == "__main__":
    main()